{>
  This program EARTH.EXE (then copied to EARTH.SCR) is a small screen
  saver example. It displays a moving earth in the sky. The EARTH.EXE
  program can also be run as a normal Windows application; in that case,
  the earth will move in a variable size window, as determined by the
  user.

  The same main form, 'f_ScreenSaver', is here used to run the screen
  saver in four different modes: execution (screen saver or normal
  Windows application) and configuration (from Windows screen saver
  installation option or normal Windows application).

  When the program runs as a normal Windows application, clicking in the
  main window will call the installation screen. In that latter mode,
  the application's icon will change every second when minimized as an
  icon.

  When the screen saver is executed, a little trick is used to prepare
  a surface on which we will draw the moving earth: the main window panel,
  'BackPanel', is made invisible and the main window is resized to use
  all the screen surface. In that case, we can then copy a earth bitmap on
  the form canvas, and all other controls that are only useful when the
  program is run in configuration mode are hidden.

  The code contains everything you need to display the configuration
  window and save the parameters to a file. These parameters are
  read back at execution time.

  Note that you need a $D directive in the main project source file,
  such as 'EARTH.DPR'. It is in this file that the .EXE name is set to
  'Earth'. It is also in the .DPR file that we have to specify
  to NOT execute the program twice (Windows may start the Screen Saver
  even when it is active).

  To install, rebuild the EARTH application, then copy the resulting
  file 'EARTH.EXE' to your Windows directory ('C:\WINDOWS', say)
  under the name 'EARTH.SCR', then use the Configuration Panel (Desktop)
  to select and possibly configure this new screen saver. You can also
  run the normal .EXE file as a normal Windows application.

  All the code is documented below, so modifying it should be very easy.
  My own comments are introduced with '{>'.

  This code is freeware. However, you can always send me an e.mail to let
  me know that you got it. It is always nice to know that something we
  contributed is actually used...! All the very best and have fun.

  Jacques Lemieux
  Bluga Soft Information, Montral, Qubec
  Compuserve: 72470,1055
  Internet: 72470.1055@compuserve.com
}

unit Earth01;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls ;

type
  Tf_ScreenSaver = class(TForm)
    BackPanel: TPanel;
    SaveButton: TButton;
    TimerEarth: TTimer;
    Earth: TImage;
    FastCheckbox: TCheckBox;
    ShowSkyCheckBox: TCheckBox;
    UseColorsCheckBox: TCheckBox;
    UseMoreStarsCheckBox: TCheckBox;
    TransparentCheckBox: TCheckBox;
    PR1: TLabel;
    PR2: TLabel;
    ExitButton: TButton;
    Icon1: TImage;
    Icon2: TImage;
    Icon3: TImage;
    Icon4: TImage;
    TimerIcon: TTimer;
    procedure FormMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure FormCreate(Sender: TObject);
    procedure TimerEarthTimer(Sender: TObject);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure SaveButtonClick(Sender: TObject);
    procedure ExitButtonClick(Sender: TObject);
    procedure ShowSkyCheckBoxClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormPaint(Sender: TObject);
    procedure FormClick(Sender: TObject);
    procedure StartSaver ;
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure TimerIconTimer(Sender: TObject);
  private
    { Private declarations }
    {> Read, Write the parameters block }
    function WinDir : string ;
    procedure read_config ;
    procedure write_config ;
    {> Setup procedure }
    procedure Setup ;
    {> Check consistency of checkboxes }
    procedure check_checkboxes ;
    {> Compute move factor }
    function move_factor : integer ;
    {> Stars handling stuff for execution mode }
    function StarColor : TColor ;
    procedure ComputeSky ;
 public
    { Public declarations }
  end;

var
  f_ScreenSaver: Tf_ScreenSaver;

implementation

{$R *.DFM}

{> Transform all I/O errors in exceptions
}
{$I+}

{> Constants & Types }

const INIT_FILE = 'M_EARTH.DAT' ;       {> Name of parameters file }

type RunMode = (rm_ScreenSaver, rm_ConfigWindows,
                rm_ConfigNormal, rm_Normal) ;

     config_params = record             {> screen saver parameters }
                       fast : boolean ;
                       show_sky : boolean ;
                       use_colors : boolean ;
                       use_more_stars : boolean ;
                       transparent : boolean ;
                       x, y : integer ;
                       width, height : integer ;
                     end ;

{> Global variables }

var gl_params : config_params ;    {> Configuration parameters }
    gl_old_mouse_pos : TPoint ;    {> Mouse position at beginning }
    gl_mode : RunMode ;            {> Current running mode }
    gl_original_width,             {> Original width, height as designed }
    gl_original_height : integer ;
    gl_icons : array [1..4] of TIcon ; {> Icons used when form minimized }
    gl_next_icon : integer ;       {> Next icon number to use }

procedure Tf_ScreenSaver.SaveButtonClick(Sender: TObject);
begin
  {> Called from configuration panel: save parameters and go back to
     execution mode as a normal application
  }
  write_config ;
  gl_mode := rm_Normal ;
  setup ;
end;

procedure Tf_ScreenSaver.ExitButtonClick(Sender: TObject);
begin
  {> Called from configuration panel: exit the program
  }
  Close ;
end;

procedure Tf_ScreenSaver.check_checkboxes ;
begin
  {> Make sure checkboxes are displayed appropriately
  }
  UseColorsCheckbox.visible := ShowSkyCheckbox.checked ;
  UseMoreStarsCheckbox.visible := ShowSkyCheckbox.checked ;
  TransparentCheckbox.visible := ShowSkyCheckbox.checked ;
end ;

procedure Tf_ScreenSaver.ShowSkyCheckBoxClick(Sender: TObject);
begin
  {> Adjust other checkboxes when this one changes status from
     checked to unchecked or vice-versa
  }
  check_checkboxes ;
end;

{> Routines to read and write the configuration parameters block
}

function Tf_ScreenSaver.WinDir : string ;
var WindowsDirectory : string [100] ;
    length : integer ;
begin
  length := GetWindowsDirectory (@WindowsDirectory [1], 100) ;
  if length = 0 then halt ;
  WindowsDirectory [0] := chr (length) ;
  WinDir := WindowsDirectory + '\' ;
end ;

procedure Tf_ScreenSaver.read_config ;
var f : file of config_params ;
begin
  {> Read back parameters file
  }
  AssignFile (f, WinDir + INIT_FILE) ;
  try
    Reset(f) ;
    Read (f, gl_params) ;
    CloseFile (f) ;
  except
    {> OOps, file not there or wrong record size, so use
       default values for the parameters instead
    }
    gl_params.fast := False ;
    gl_params.show_sky := True ;
    gl_params.use_colors := False ;
    gl_params.use_more_stars := False ;
    gl_params.transparent := False ;
    {> Params used in normal mode only
    }
    gl_params.x := 50 ;
    gl_params.y := 50 ;
    gl_params.width := earth.width * 2 ;
    gl_params.height := earth.height * 2 ;
  end ;
end ;

procedure Tf_ScreenSaver.write_config ;
var f : file of config_params ;
begin
  {> Write back current parameters to file
  }
  if gl_mode in [rm_ConfigWindows, rm_ConfigNormal] then
  begin
    gl_params.fast := FastCheckbox.checked ;
    gl_params.show_sky := ShowSkyCheckbox.checked ;
    gl_params.use_colors := UseColorsCheckbox.checked ;
    gl_params.use_more_stars := UseMoreStarsCheckbox.checked ;
    gl_params.transparent := TransparentCheckbox.checked ;
  end ;

  if gl_mode = rm_Normal then
  begin
    gl_params.x := Left ;
    gl_params.y := Top ;
    gl_params.width := ClientWidth ;
    gl_params.height := ClientHeight ;
  end ;

  AssignFile (f, WinDir + INIT_FILE) ;
  Rewrite (f) ;
  Write (f, gl_params) ;
  CloseFile (f) ;
end ;

procedure Tf_ScreenSaver.Setup ;
begin
  {> Make the earth bitmap full size
  }
  earth.AutoSize := True ;

  TimerEarth.Enabled := False ;

  read_config ;

  visible := False ;

  if gl_mode in [rm_ConfigWindows, rm_ConfigNormal] then
  begin
    {> If running in install mode, use normal form
    }
    BorderStyle := bsDialog ;
    FormStyle := fsNormal ;
    Width := gl_original_width ;
    Height := gl_original_Height ;
    BackPanel.Visible := True ;
    position := poScreenCenter ;
    SaveButton.visible := gl_mode = rm_ConfigNormal ;

    {> Fill in parameters on the screen...
    }
    FastCheckbox.checked := gl_params.fast ;
    ShowSkyCheckbox.checked := gl_params.show_sky ;
    UseColorsCheckbox.checked := gl_params.use_colors ;
    UseMoreStarsCheckbox.checked := gl_params.use_more_stars ;
    TransparentCheckbox.checked := gl_params.transparent ;
    check_checkboxes ;

    {> Make sure the earth bitmap is not visible in this case
    }
    earth.visible := False ;
  end
  else
  begin
    {> If running in execution mode, use a form with no borders,
       make sure it is 'StayOnTop', and resize it to use the whole
       screen.
    }
    if gl_mode = rm_Normal then
    begin
      caption := 'Earth' ;
      FormStyle := fsNormal ;
      BorderStyle := bsSizeable ;
      Left := gl_params.x ;
      Top := gl_params.y ;
      ClientWidth := gl_params.width ;
      ClientHeight := gl_params.height ;
      position := poDesigned ;
    end
    else
    begin
      BorderStyle := bsNone ;
      FormStyle := fsStayOnTop ;
      Width := Screen.Width ;
      Height := Screen.Height ;
      Cursor := crNone ;
      Top := 0 ;
      Left := 0 ;
    end ;

    {> Make the main window panel invisible, so this will zap out
       all the controls that have been put on it.
    }
    BackPanel.Visible := False ;

    {> Start the screen saver as such
    }
    StartSaver ;
  end ;

  visible := True ;
end;


procedure Tf_ScreenSaver.FormCreate(Sender: TObject);
begin
  {> Set up array for dynamic icons (used when minimized)
  }
  gl_next_icon := 1 ;
  gl_icons [1] := Icon1.Picture.Icon ;
  gl_icons [2] := Icon2.Picture.Icon ;
  gl_icons [3] := Icon3.Picture.Icon ;
  gl_icons [4] := Icon4.Picture.Icon ;
  Icon1.visible := false ;
  Icon2.visible := false ;
  Icon3.visible := false ;
  Icon4.visible := false ;

  {> Save initial size as designed
  }
  gl_original_width := width ;
  gl_original_height := height ;

  {> Determine run mode
  }
  if ParamStr(1) = '/c' then
    gl_mode := rm_ConfigWindows
  else
  if ParamStr(1) = '/s' then
    gl_mode := rm_ScreenSaver
  else
    gl_mode := rm_Normal ;

  {> Continue setup
  }
  setup ;
end ;

procedure Tf_ScreenSaver.FormMouseMove(Sender: TObject;
                                       Shift: TShiftState; X, Y: Integer);
begin
  {> If running in normal Windows application mode, do nothing
  }
  if gl_mode = rm_Normal then
    exit ;

  {> If called the first time, remember mouse initial position, else
     terminate if the mouse has really moved.
  }
  if gl_old_mouse_pos.X = -1 then
    gl_old_mouse_pos := Point (X, Y)
  else
  if (gl_old_mouse_pos.X <> X) or (gl_old_mouse_pos.Y <> Y) then
    Close ;
end;

procedure Tf_ScreenSaver.FormKeyPress(Sender: TObject; var Key: Char);
begin
  {> If running in normal Windows application mode, do nothing
  }
  if gl_mode = rm_Normal then
    exit ;

  {> Terminate when a key is pressed.
  }
  Close ;
end;

{> This is the screen saver code as such, which moves a bitmap
   around. This code is no longer (almost) generic.
}

const N_STARS = 500 ;
      N_BUMPS = 15 ;

type TStar = record                        {> To describe a star }
               x0, y0, x1, y1 : integer ;  {> Area covered by star }
               xc, yc : real ;             {> Center of star }
               radius : real ;             {> Radius of star }
               color : TColor ;            {> And its color }
             end ;

var gl_vert, gl_horiz,                     {> Move factors }
    gl_top, gl_left : integer ;            {> Current earth position }
    gl_n_stars : integer ;                 {> Number of stars displayed }
    gl_stars : array [1..N_STARS] of       {> Stars definition }
               TStar ;
    gl_n_bumps_left : integer ;            {> Bumps left before new sky }

procedure Tf_ScreenSaver.StartSaver ;
var b : TBitmap ;
begin
  {> Make sure the space around the earth contains only pure black color;
     looks nicer when the color palette is modified...
  }
  b := TBitmap.Create ;
  b.monochrome := false ;
  b.width := Earth.width ;
  b.height := Earth.height ;
  b.canvas.brush.style := bsSolid ;
  b.canvas.brush.color := clBlack ;
  b.canvas.FillRect (Rect (0, 0, b.Width, b.Height)) ;
  b.canvas.brush.color := clWhite ;
  b.canvas.Ellipse (0, 0, b.Width, b.Height) ;
  Earth.picture.bitmap.canvas.CopyMode := cmSrcAnd ;
  Earth.picture.bitmap.canvas.draw (0, 0, b) ;
  b.free ;

  {> Initialize flags to ensure that the phoney mouse move events
     will be ignored.
  }
  gl_old_mouse_pos := Point (-1, -1) ;

  {> Start random numbers generation
  }
  randomize ;

  {> Compute initial location for earth
  }
  gl_left := Random (ClientWidth - Earth.Width) ;
  gl_top := Random (ClientHeight - Earth.Height) ;

  {> Set initial move directions
  }
  gl_vert := move_factor ;
  gl_horiz := move_factor ;
  if random (2) = 0 then
    gl_vert := -gl_vert ;
  if random (2) = 0 then
    gl_horiz := -gl_horiz ;

  {> Start timer to do the earth bitmap moves on a regular basis
  }
  TimerEarth.Interval := 80 ;
  if gl_params.fast then
    TimerEarth.Interval := TimerEarth.Interval div 2;
  TimerEarth.Enabled := True ;

  {> Compute number of stars required (if full screen used)
  }
  gl_n_stars := N_STARS ;
  if not gl_params.use_more_stars then
    gl_n_stars := gl_n_stars div 2 ;

  {> Compute number of stars according to current real screen area used
  }
  gl_n_stars := round ((1.0 * ClientHeight * ClientWidth) /
                       (1.0 * Screen.Height * Screen.Width) *
                       gl_n_stars) ;

  {> Make sure the sky will be displayed at the first iteration
  }
  gl_n_bumps_left := 0 ;
end ;

function Tf_ScreenSaver.StarColor : TColor ;
var color : TColor ;
begin
  {> Determine the color of a star on a random basis
  }
  if gl_params.use_colors then
    case random (40) of
      0..10 :   color := clSilver ;
      11..30 :  color := clWhite ;
      31 :      color := clRed ;
      32..37 :  color := clYellow ;
      38 :      color := clLime ;
      39 :      color := clTeal ;
    end
  else
  if random (2) = 0 then
    color := clWhite
  else
    color := clSilver ;
  StarColor := color ;
end ;

procedure Tf_ScreenSaver.ComputeSky ;
var i, size : integer ;
begin
  {> Compute positions for stars, their radius, etc...
  }
  for i := 1 to gl_n_stars do
    with gl_stars [i] do
    begin
      x0 := random (ClientWidth) ;
      y0 := random (ClientHeight) ;
      if random (40) = 0 then
        size := 3 + random (5)
      else
        size := random (2) + 1 ;
      x1 := x0 + size ;
      y1 := y0 + size ;
      xc := (x0 + x1) / 2.0 ;
      yc := (y0 + y1) / 2.0 ;
      radius := (x1 - x0 + 1) / 2.0 ;
      color := StarColor ;
    end ;
end ;

function Tf_ScreenSaver.move_factor : integer ;
begin
  {> Compute delta move factor. We want to move by one pixel
     only most of the time. Once in a while, use a different
     value to make sure that the earth does not enter a static
     bumping behavior...
  }
  move_factor := 1 + ord (random (8) = 0) ;
end ;

procedure Tf_ScreenSaver.TimerEarthTimer(Sender: TObject);
var old_top, old_left : integer ;
    b : TBitmap ;
    bumping : boolean ;
    fx0, fy0, fx1, fy1, x, y : integer ;
    i : integer ;
    in_h, in_v : boolean;
    distance, radius_earth : real ;
    n_blink, k : integer ;
begin
  {> Procedure called to move the earth bitmap around. If iconic, do
     nothing...
  }
  if IsIconic (Application.handle) then
    exit ;

  {> Save current earth coordinates
  }
  old_top := gl_top ;
  old_left := gl_left ;

  {> Increase bitmap position
  }
  gl_top := gl_top + gl_vert ;
  gl_left := gl_left + gl_horiz ;

  {> Check if we bumped on the borders; if so, change direction
  }
  bumping := false ;
  if gl_top < 0 then
  begin
    gl_vert := move_factor ;
    gl_top := gl_vert ;
    bumping := true ;
  end
  else
  if gl_top + earth.height > ClientHeight then
  begin
    gl_vert := - move_factor ;
    gl_top := ClientHeight - earth.Height ;
    bumping := true ;
  end ;

  if gl_left < 0 then
  begin
    gl_horiz := move_factor ;
    gl_left := gl_horiz ;
    bumping := true ;
  end
  else
  if gl_left + earth.width > ClientWidth then
  begin
    gl_horiz := - move_factor ;
    gl_left := ClientWidth - earth.Width ;
    bumping := true ;
  end ;

  {> Compute coordinates of the area about to be modified on the screen
     (fx0, fy0) --> (fx1, fy1).
  }
  if old_left < gl_left then
  begin
    fx0 := old_left ;
    fx1 := gl_left + earth.width
  end
  else
  begin
    fx0 := gl_left ;
    fx1 := old_left + earth.width ;
  end ;

  if old_top < gl_top then
  begin
    fy0 := old_top ;
    fy1 := gl_top + earth.height ;
  end
  else
  begin
    fy0 := gl_top ;
    fy1 := old_top + earth.height ;
  end ;

  {> Add some buffer zone to make sure we include stars that may fall
     just outside of the frame, but have a portion in that frame
  }
  dec (fx0, 8) ;
  inc (fx1, 8) ;
  dec (fy0, 8) ;
  inc (fy1, 8) ;

  {> Create a bitmap of these dimensions and blank it.
  }
  b := TBitmap.Create ;
  b.monochrome := false ;
  b.width := fx1 - fx0 + 1 ;
  b.height := fy1 - fy0 + 1 ;
  b.canvas.brush.style := bsSolid ;
  b.canvas.brush.color := clBlack ;
  b.canvas.fillrect (Rect (0, 0, b.Width, b.Height)) ;

  {> Draw earth bitmap at new position
  }
  b.canvas.Draw (gl_left - fx0, gl_top - fy0, earth.picture.bitmap) ;

  {> Draw stars in the modified area, but not over the earth if not
     transparent
  }
  if gl_params.show_sky then
  begin
    {> Check if time to redisplay the background
    }
    if gl_n_bumps_left = 0 then
      bumping := true ;
    if bumping then
      if gl_n_bumps_left = 0 then
      begin
        ComputeSky ;
        FormPaint (Nil) ;
        gl_n_bumps_left := N_BUMPS ;
      end
      else
        dec (gl_n_bumps_left) ;

    {> Make some stars blink a bit because it is nice. Ignore stars
       that fall in the modified frame because it is too complicated
       to handle them here.
    }
    n_blink := random (gl_n_stars div 10) ;
    for i := 1 to n_blink do
    begin
      k := random (gl_n_stars) + 1 ;
      with gl_stars [k] do
        if (x1 < fx0) or (x0 > fx1) or
           (y1 < fy0) or (y0 > fy1) then
        begin
          color := StarColor ;
          canvas.brush.color := color ;
          canvas.pen.color := color ;
          canvas.pen.mode := pmCopy ;
          canvas.pen.width := 1 ;
          canvas.ellipse (x0, y0, x1, y1) ;
        end ;
    end ;

    {> Compute radius and middle coordinates of earth
    }
    radius_earth := earth.width / 2.0 ;
    x := gl_left + earth.width div 2 ;
    y := gl_top + earth.height div 2 ;

    {> Add stars that fall inside our modified area
    }
    for i := 1 to gl_n_stars do
      with gl_stars [i] do
      begin
        in_h := (x0 >= fx0) and (x0 <= fx1) or
                (x1 >= fx0) and (x1 <= fx1) ;
        if not in_h then
          continue ;
        in_v := (y0 >= fy0) and (y0 <= fy1) or
                (y1 >= fy0) and (y1 <= fy1) ;
        if not in_v then
          continue ;
        distance := sqrt (sqr (1.0 * (xc - x)) +
                          sqr (1.0 * (yc - y))) ;
        if (distance - radius > radius_earth) or
           gl_params.transparent then
          with b.canvas do
          begin
            brush.color := color ;
            pen.color := color ;
            pen.mode := pmCopy ;
            pen.width := 1 ;
            ellipse (x0 - fx0, y0 - fy0, x1 - fx0, y1 - fy0) ;
          end ;
      end ;
  end ;

  {> Output the final bitmap and then free it
  }
  canvas.draw (fx0, fy0, b) ;

  b.free ;
end;

procedure Tf_ScreenSaver.FormResize(Sender: TObject);
begin
  {> If not running in normal mode, ignore
  }
  if gl_mode <> rm_Normal then
    exit ;

  {> Make sure new dimensions are ok to at least include the earth and a
     bit of room to move
  }
  if ClientWidth < 1.25 * earth.width then
    ClientWidth := round (1.25 * earth.width) ;
  if ClientHeight < 1.25 * earth.Height then
    ClientHeight := round (1.25 * earth.height) ;

  {> Restart the screen saver with these new dimensions
  }
  FormPaint (Nil) ;
  StartSaver ;
end;

procedure Tf_ScreenSaver.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  {> Save config whenever the program terminates
  }
  write_config ;
end;

procedure Tf_ScreenSaver.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  {> Save config whenever the program terminates following a request
     from Windows (end of Windows session, for example).
  }
  CanClose := true ;
  write_config ;
end;

procedure Tf_ScreenSaver.FormPaint(Sender: TObject);
begin
  {> Redisplay the background when window has to be repainted
  }
  canvas.brush.color := clBlack ;
  canvas.FillRect (Rect (0, 0, ClientWidth, ClientHeight)) ;
end;

procedure Tf_ScreenSaver.FormClick(Sender: TObject);
begin
  {> If not running in normal mode, do nothing
  }
  if gl_mode <> rm_Normal then
    exit ;

  {> Else, run the config. panel
  }
  write_config ;
  gl_mode := rm_ConfigNormal ;
  setup ;
end;

procedure Tf_ScreenSaver.TimerIconTimer(Sender: TObject);
begin
  {> When application running in normal mode and minimized, change the
     application every second, just for fun.
  }
  if not IsIconic (Application.Handle) then
    exit ;

  Application.Icon := gl_icons [gl_next_icon] ;
  gl_next_icon := gl_next_icon mod 4 + 1 ;

  {> Thanks to Tim Noonan for these two little lines that ask Windows
     to refresh the icon bitmap.
  }
  InvalidateRect (Application.handle, Nil, True) ;
  UpdateWindow (Application.handle) ;
end;

{> That's it, folks.
}
end.
